/******************************************************************
 *FILE: synaptics_intr.c
 *SW-COMPONENT: QNX synaptics touch driver
 *DESCRIPTION: This source file is to configure GPIO and interrupt
 *for atmel touch device.
 *COPYRIGHT: © 2019 Robert Bosch GmbH
 *
 *This program is free software; you can redistribute  it and/or modify it
 *under  the terms of  the GNU General  Public License as published by the
 *Free Software Foundation;  either version 2 of the  License, or (at your
 *option) any later version.
 ******************************************************************/
#include <string.h>
#include <errno.h>
#ifndef UNITTEST
#include <input/mtouch_log.h>
#include <sys/neutrino.h>
#include <gpio_client.h>
#else
#include "gpio_header.h"
#endif

#include "syna_tcm2_mtouch.h"

unsigned short syna_mtouch_gpio_conf(struct syna_tcm_dev *p_dev)
{
    uint32_t gpio_number = p_dev->hw_if->bdata_attn.irq_gpio;
    uint32_t gpio_number_tlmm;
    uint32_t gpio_config;
    int fd;

    struct syna_hw_attn_data *attn;
    attn = &p_dev->hw_if->bdata_attn;

    if (-1 == (fd = gpio_open(NULL))) {
        LOGE("gpio_open() failed");
        error_memory("Synaptics_Touch: gpio_open() failed");
        goto exit_err;
    }

    if(gpio_number  <= 0 ){
        LOGE("Bad GPIO input param gpio=%d \n",gpio_number);
        error_memory("Synaptics_Touch: Bad GPIO input param gpio=%d \n",gpio_number);
        goto exit_err;
    }

    gpio_number_tlmm = gpio_number & 0xFFFF; // for TLMM upper 16 bit mask is 0s
    gpio_config = GPIO_PIN_CFG(GPIO_INPUT,GPIO_PULL_UP,GPIO_STRENGTH_2MA,0);

    /* set gpio configuration for the gpio */
    if (GPIO_SUCCESS != gpio_set_config(fd, gpio_number_tlmm, 0, gpio_config)){
        LOGE("gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        error_memory("Synaptics_Touch: gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        goto exit_err;
    }

    /* set interrupt cfg for the gpio */
    if (GPIO_SUCCESS != gpio_set_interrupt_cfg(fd, gpio_number_tlmm, GPIO_INTERRUPT_TRIGGER_LOW, NULL)){
        LOGE("gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        error_memory("Synaptics_Touch: gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        goto exit_err;
    }

    /* get interrupt irq number from the configured interrupt gpio */
    if (GPIO_SUCCESS != gpio_get_interrupt_cfg(fd, gpio_number_tlmm, &attn->tp_intr)){
        LOGE("failed to get irq for gpio gpio_number %d \n", gpio_number);
        error_memory("Synaptics_Touch: failed to get irq for gpio gpio_number %d \n", gpio_number);
        goto exit_err;
    }

    LOGI("irq corresponding to gpio %d is %d - exiting\n", gpio_number, attn->tp_intr);

    return 0;
exit_err:
    return -1;
}

int
syna_mtouch_tp_intr_attach(struct syna_tcm_dev *p_dev)
{
    int ret;
    struct syna_hw_attn_data *attn;
    attn = &p_dev->hw_if->bdata_attn;

    ret = syna_mtouch_gpio_conf(p_dev);
    if (ret == -1) {
        LOGE("failed to set gpio configuration \n ");
        error_memory("Synaptics_Touch: failed to set gpio configuration \n ");
        goto fail1;
    }

    SIGEV_PULSE_INIT(&p_dev->thread_event, p_dev->thread_coid, PULSE_CODE_PRIORITY, PULSE_CODE, 0);

    attn->irq_id = InterruptAttachEvent(attn->tp_intr, &p_dev->thread_event, _NTO_INTR_FLAGS_TRK_MSK | _NTO_INTR_FLAGS_PROCESS);
    if (attn->irq_id == -1) {
        LOGE("InterruptAttach: %s", strerror(errno));
        error_memory("Synaptics_Touch: InterruptAttach: %s", strerror(errno));
        goto fail1;
    }

    attn->irq_enabled = true;
    LOGI("Attached to device with interrupt %d", attn->tp_intr);

    return 0;

fail1:
    return -1;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/atmel/atmel_tp_intr.c $ $Rev: 809524 $")
#endif
